<?php

namespace App\Controllers;

use App\Models\EstudianteModel;
use CodeIgniter\RESTful\ResourceController;

class EstudiantesController extends ResourceController
{
    protected $format = 'json';
    
    public function index()
    {
        $model = new EstudianteModel();
        
        // Búsqueda opcional
        $search = $this->request->getGet('search');
        
        if ($search) {
            $estudiantes = $model->buscarEstudiantes($search);
        } else {
            $estudiantes = $model->where('estado', 'activo')->findAll();
        }
        
        return $this->respond([
            'success' => true,
            'message' => 'Estudiantes obtenidos correctamente',
            'data' => $estudiantes
        ]);
    }
    
    public function show($id = null)
    {
        $model = new EstudianteModel();
        
        $estudiante = $model->find($id);
        
        if ($estudiante) {
            return $this->respond([
                'success' => true,
                'message' => 'Estudiante encontrado',
                'data' => $estudiante
            ]);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Estudiante no encontrado',
            'data' => null
        ], 404);
    }
    
    public function create()
    {
        $model = new EstudianteModel();
        
        $json = $this->request->getJSON();
        
        if (!$json) {
            return $this->respond([
                'success' => false,
                'message' => 'Datos inválidos',
                'data' => null
            ], 400);
        }
        
        $data = [
            'codigo_estudiante' => $json->codigo_estudiante ?? '',
            'nombres' => $json->nombres ?? '',
            'apellidos' => $json->apellidos ?? '',
            'email' => $json->email ?? '',
            'telefono' => $json->telefono ?? null,
            'direccion' => $json->direccion ?? null
        ];
        
        // Validaciones básicas
        if (empty($data['codigo_estudiante']) || empty($data['nombres']) || empty($data['apellidos'])) {
            return $this->respond([
                'success' => false,
                'message' => 'Código, nombres y apellidos son requeridos',
                'data' => null
            ], 400);
        }
        
        // Verificar código único
        if ($model->where('codigo_estudiante', $data['codigo_estudiante'])->first()) {
            return $this->respond([
                'success' => false,
                'message' => 'El código de estudiante ya existe',
                'data' => null
            ], 409);
        }
        
        if ($model->insert($data)) {
            $estudiante = $model->find($model->getInsertID());
            
            return $this->respond([
                'success' => true,
                'message' => 'Estudiante creado exitosamente',
                'data' => $estudiante
            ], 201);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al crear estudiante',
            'data' => null
        ], 500);
    }
    
    public function update($id = null)
    {
        $model = new EstudianteModel();
        
        $estudiante = $model->find($id);
        
        if (!$estudiante) {
            return $this->respond([
                'success' => false,
                'message' => 'Estudiante no encontrado',
                'data' => null
            ], 404);
        }
        
        $json = $this->request->getJSON();
        
        $data = [
            'nombres' => $json->nombres ?? $estudiante['nombres'],
            'apellidos' => $json->apellidos ?? $estudiante['apellidos'],
            'email' => $json->email ?? $estudiante['email'],
            'telefono' => $json->telefono ?? $estudiante['telefono'],
            'direccion' => $json->direccion ?? $estudiante['direccion'],
            'estado' => $json->estado ?? $estudiante['estado']
        ];
        
        if ($model->update($id, $data)) {
            $estudianteActualizado = $model->find($id);
            
            return $this->respond([
                'success' => true,
                'message' => 'Estudiante actualizado exitosamente',
                'data' => $estudianteActualizado
            ]);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Error al actualizar estudiante',
            'data' => null
        ], 500);
    }
    
    public function delete($id = null)
    {
        $model = new EstudianteModel();
        
        if ($model->find($id)) {
            // Soft delete
            $model->update($id, ['estado' => 'inactivo']);
            
            return $this->respond([
                'success' => true,
                'message' => 'Estudiante eliminado exitosamente',
                'data' => null
            ]);
        }
        
        return $this->respond([
            'success' => false,
            'message' => 'Estudiante no encontrado',
            'data' => null
        ], 404);
    }
}
